<?php

namespace App\Http\Controllers;

use App\Http\Controllers\AppBaseController;

use App\Models\KanbanAccounts;
use App\Models\BoardSwimlanes;
use App\Models\BoardStages;
use App\Models\Boards;
use App\Models\GlobalSSMappings;
use App\Models\GlobalSSRelations;
use App\Models\GlobalSSMappingLevels;

use Illuminate\Http\Request;
use Flash;
use Response;

class GlobalSSMappingController extends AppBaseController
{
    /**
     * Display a listing of the Swimline Mapping.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function index(Request $request)
    {
        $GlobalSSMappings = GlobalSSMappings::get();

        foreach($GlobalSSMappings as $gsm)
        {
            foreach($gsm->global_ss_relations as $gsr)
            {
                if($gsr->swimlane_ids != "")
                {
                    $gsr->swimlane_names = BoardSwimlanes::whereIn("swimlane_id", explode(',', $gsr->swimlane_ids))->pluck('name')->implode(', ');
                }

                if($gsr->stage_ids != "")
                {
                    $gsr->stage_names = BoardStages::whereIn("stage_id", explode(',', $gsr->stage_ids))->pluck('name')->implode(', ');
                }

            }

        }

        return view('global_ss_mapping.index')
            ->with('GlobalSSMappings', $GlobalSSMappings);

    }

    /**
     * Show the form for creating a new KanbanAccounts.
     *
     * @return Response
     */
    public function create()
    {
        $boards = Boards::get();

        $levels = GlobalSSMappingLevels::whereIn("id", GlobalSSMappings::pluck("level_id"))->get();

        foreach($boards as $board)
        {
           $tree = $this->buildTree($board->stages->toArray());

           $board->tree_stages = $tree;
        }
        
        return view('global_ss_mapping.create')->with('boards', $boards)->with('levels', $levels);
    }

    public function buildTree(array $elements, $parentId = 0) 
    {
        $branch = array();

        foreach ($elements as $element) {
            if ($element['parent_id'] == $parentId) {
                $children = $this->buildTree($elements, $element['stage_id']);
                if ($children) {
                    $element['children'] = $children;
                }
                $branch[] = $element;
            }
        }

        return $branch;
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'level_select' => 'required_without:level_new',
            'level_new' => 'required_without:level_select'
        ]);
        
        if($request->level_new != "")
        {
            $g_ss_map_level = GlobalSSMappingLevels::create(["label" => $request->level_new]);
            $level_id = $g_ss_map_level->id;
        }else
        {
            $level_id = $request->level_select;
        }

        $g_ss_map = GlobalSSMappings::create(["level_id" => $level_id, "label" => $request->label]);

        $g_ss_map_relations = [];
        foreach($request->boards as $b)
        {
            $g_ss_map_relations_single = [];
            $g_ss_map_relations_single['global_ss_mapping_id'] = $g_ss_map->id;
            $g_ss_map_relations_single['board_id'] = $b;

            if(isset($request->swimlanes[$b]) && count($request->swimlanes[$b]) > 0)
            {
                $g_ss_map_relations_single['swimlane_ids'] = implode(',',$request->swimlanes[$b]);
            }
            if(isset($request->stages[$b]) && count($request->stages[$b]) > 0)
            {
                $g_ss_map_relations_single['stage_ids'] = implode(',',$request->stages[$b]);
            }

            if(isset($request->swimlanes[$b]) && isset($request->stages[$b]))
            {
                array_push($g_ss_map_relations, $g_ss_map_relations_single);
            }
            
            
        }

        if(count($g_ss_map_relations) > 0)
        {
            GlobalSSRelations::insert($g_ss_map_relations);
        }

        return redirect(route('global_ss_mapping.index'));

    }

    public function edit($id)
    {
        $GlobalSSMappings = GlobalSSMappings::find($id);

        $boards = Boards::get();

        $levels = GlobalSSMappingLevels::whereIn("id", GlobalSSMappings::pluck("level_id"))->get();

        foreach($boards as $board)
        {
           $tree = $this->buildTree($board->stages->toArray());

           $board->tree_stages = $tree;

           $selected_items = GlobalSSRelations::where("global_ss_mapping_id", $id)->where("board_id", $board->id)->select("swimlane_ids", "stage_ids")->first();

           if($selected_items)
           {
                $board->selected_swimlanes = explode(",", $selected_items->swimlane_ids);
                $board->selected_stages = explode(",", $selected_items->stage_ids);
           }
           else
           {
            $board->selected_swimlanes = [];
            $board->selected_stages = [];
           }
        }

        if (empty($GlobalSSMappings)) {
            Flash::error('Global Snapshot Mapping not found');

            return redirect(route('global_ss_mapping.index'));
        }
        
        return view('global_ss_mapping.edit')->with('GlobalSSMappings', $GlobalSSMappings)->with('boards', $boards)->with('levels', $levels);
    }

    public function update($id, Request $request)
    {
        $validated = $request->validate([
            'level_select' => 'required_without:level_new',
            'level_new' => 'required_without:level_select'
        ]);

        if($request->level_new != "")
        {
            $g_ss_map_level = GlobalSSMappingLevels::create(["label" => $request->level_new]);
            $level_id = $g_ss_map_level->id;
        }else
        {
            $level_id = $request->level_select;
        }

        $GlobalSSMappings = GlobalSSMappings::find($id);
        $GlobalSSMappings->level_id = $level_id;
        $GlobalSSMappings->label = $request->label;
        $GlobalSSMappings->save();

        $g_ss_map_relations = [];
        foreach($request->boards as $b)
        {
            $g_ss_map_relations_single = [];
            $g_ss_map_relations_single['global_ss_mapping_id'] = $id;
            $g_ss_map_relations_single['board_id'] = $b;

            if(isset($request->swimlanes[$b]) && count($request->swimlanes[$b]) > 0)
            {
                $g_ss_map_relations_single['swimlane_ids'] = implode(',',$request->swimlanes[$b]);
            }
            if(isset($request->stages[$b]) && count($request->stages[$b]) > 0)
            {
                $g_ss_map_relations_single['stage_ids'] = implode(',',$request->stages[$b]);
            }

            if(isset($request->swimlanes[$b]) && isset($request->stages[$b]))
            {
                array_push($g_ss_map_relations, $g_ss_map_relations_single);
            }
            
        }

        GlobalSSRelations::where('global_ss_mapping_id', $id)->delete();

        if(count($g_ss_map_relations) > 0)
        {
            GlobalSSRelations::insert($g_ss_map_relations);
        }

        Flash::success('Global Snapshot Mapping updated successfully.');

        return redirect(route('global_ss_mapping.index'));

    }

    public function destroy($id)
    {
        /** @var GlobalSSMappings $GlobalSSMappings */
        $GlobalSSMappings = GlobalSSMappings::find($id);

        if (empty($GlobalSSMappings)) {
            Flash::error('Mapping not found');

            return redirect(route('global_ss_mapping.index'));
        }

        $GlobalSSMappings->delete();

        Flash::success('Mapping deleted successfully.');

        return redirect(route('global_ss_mapping.index'));
    }
}